%
% BrownAtam_StateAwareMCMC.m
%
% D. Andrew Brown and Garrison Stevens
% 25 November 2014
%
% This code implements an adaptive MH algorithm for simulating the 
% posterior of the state-aware calibration model.  
%
% Most recent update: 5 Januray 2016


            
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%load('simDataTX2_ObsData.mat')

% xObs are the observed control inputs, and yObs is the observed field data
%pts= 1:5:100;
%x= x(pts);
%xObs= x([1:9 15:20]);
%yObs= yObs(pts);

% A little pre-processing of the data
yObsFull= yObs;  % Save the full dataset before redefining it
yObs= yObs([1:9 15:20]);

% Location/Scale for field data / eta output
FldAvg= mean(yObs);
FldSD= 0.05;

% Location/Scale for control inputs (x) - already on [0,1] scale

% Bounds on calibration parameters
n0Min= 2.35;
n0Max= 2.65;
tauMin= -0.5;
tauMax= 2.5;

% Adding constraints to the value that tau can take at the endpoints of the GPM.
tau1Min= (-0.075 - tauMin)/(tauMax - tauMin);
tau1Max= (0.075 - tauMin)/(tauMax - tauMin);
tau2Min= (1.85 - tauMin)/(tauMax - tauMin);
tau2Max= (2.05 - tauMin)/(tauMax - tauMin);


% Put the endpoint constraints on the logit scale ('om' = omega)
om1Min= log(tau1Min/(1-tau1Min));
om1Max= log(tau1Max/(1-tau1Max));
om2Min= log(tau2Min/(1-tau2Min));
om2Max= log(tau2Max/(1-tau2Max));

                               
yObsStd= (yObs - FldAvg)./FldSD;  % Center and scale the field data. Note
                                  % that the eta output will need to be
                                  % centered and scaled by the same amount.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



% hyperparameters
% It might be appropriate to adjust these when applying the GP on the
% logit scale.
%a_lamb= 5;  % hyperparams for tau GP precision param
%b_lamb= 5;
a_lamb= 0.01;
b_lamb= 0.01;

a_y= 5;
b_y= 5;  % hyperparams for error precision prior

b_tau= 0.2;  % hyperparam for tau GP smoothness param

yFld= yObsStd;

% Define the physics model (assume fast-running code so we can call this
% at will with minimal computational expense). The parameters are
% scaled to lie in the unit hypercube in the MCMC. 
% So etaSim here takes as inputs the scaled parameter values.
etaSim= @(a, b, loc) (((n0Max - n0Min).*b + n0Min).*(loc.^2) + ...
                        ((tauMax - tauMin).*a  + tauMin) - FldAvg)./0.05; 


%% Initialize values for each chain


% Test with a different set of starting values for everychain

nchains = 3; % select number of chains to assess
nburn = 10000; % selct number of runs for burn-in loop
nsamples = 4000; % select number of runs for sampling loop


% Choose and initial path that agrees with prior constraints on the 
% logit scale.

% Chain 1
om1= repmat(0.5, length(xObs), 1);

tau1= exp(om1)./(1 + exp(om1));  % Put the path on the original scale, as well

n01= 0.5;  % Initialize other (constant) parameter and the GP params
lamb_t1= 1;
rho_t1= 0.5;



% Chain 2
om2= repmat(0.1, length(xObs), 1);

tau2= exp(om2)./(1 + exp(om2));  % Put the path on the original scale, as well

n02= 0.75;
lamb_t2= 1.25;
rho_t2= 0.95;

% Chain 3
om3= repmat(0.9, length(xObs), 1);

tau3= exp(om3)./(1 + exp(om3));  % Put the path on the original scale, as well

n03= 0.1;
lamb_t3= 0.75;
rho_t3= 0.9;


needNewWorkers = (matlabpool('size') == 0);
if needNewWorkers
    matlabpool open
end



parfor chain_count = 1:nchains % parfor loop to parallelize chains
    if chain_count == 1
        om= om1; tau=tau1; n0=n01; lamb_t=lamb_t1; rho_t=rho_t1;
    elseif chain_count == 2
        om= om2; tau=tau2; n0=n02; lamb_t=lamb_t2; rho_t=rho_t2;
    elseif chain_count ==3
        om= om3; tau=tau3; n0=n03; lamb_t=lamb_t3; rho_t=rho_t3;
    end
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    % These will adapt periodically to tune the acceptance rates
    % while doing the burn in step
    c= 0.05;
    cN0= 0.5;
    cRho= 0.5;
    
    % Burn-in step
    acceptCount= 0;  
    acceptCountN0= 0;
    acceptCountRho= 0;
    
    
    initTau= zeros(size(tau,1), 1000);  % hold the posterior draws of tau
    initN0= zeros(1,1000);  % hold the posterior draws of n0
    %initLambT= zeros(1,1000);  % hold the posterior draws of lamb_t
    initRhoT= zeros(1,1000);  % hold the posterior draws of rho_t
    initLambY= zeros(1,1000);
    
    for iter= 1:nburn
        

        % Draw lamb_y
        a_yStar= a_y + size(tau, 1)./2;
        b_yStar= b_y + 0.5.*((yFld' - etaSim(tau, n0, xObs'))'* ...
             (yFld' - etaSim(tau, n0, xObs')));     
        lamb_y= gamrnd(a_yStar, (b_yStar).^(-1));
        %%%%%
        
        
        
        % Draw tau vector
        % initialize and Calculate upper-traingular of R
        R= zeros(size(tau,1), size(tau,1));  % Initialize the correlation
        % matrix
        for i= 1:size(R,1)
            
            for j= i:size(R,2)
                
                R(i,j)= rho_t^(4.*abs(xObs(i) - xObs(j)).^2);
                
            end
            
        end
        
        R= R + R' - diag(diag(R));  % Fill in lower traingular of R
        
        % Add a small ridge (white noise) to the diagonal of R to make
        % it "more non-singular". Use the bound provided by Ranjan
        % et al. (2011, Technometrics), using some value of a. 
        maxEig= max(eig(R));
        cndNum= cond(R);
        bnd= max([maxEig.*(cndNum - exp(20))./(cndNum.*(exp(20)-1)) 0]);
        
        delta= bnd;
        Rd= R + delta.*eye(size(R,1), size(R,1));
        
        
        
        % Draw tau vector using Metropolis proposals and block sampling
        [U, D]= eig((lamb_t).^(-1).*R);  % eigendecomposition is slower than
        % Cholesky, but more
        % computationally stable.
        
        %         if p ~= 0
        %
        %             'problem with R'
        %             break
        %
        %         end
        
        
        
        % Neal (1998) suggested doing several iterations of the GP draws for
        % the Metropolis-within-Gibbs step to improve convergence
        om_temp= om;
        temp= exp(om)./(1 + exp(om));
        for subIter= 1:5
            
            om_star= c.*U*diag(sqrt(diag(D)))*randn(size(om)) + om_temp;
            om_star= real(om_star);
            
            

            tau_star= exp(om_star)./(1 + exp(om_star));
            


            logTauStar= -0.5.*lamb_y.*(yFld' - etaSim(tau_star, n0, xObs'))'* ...
                (yFld' - etaSim(tau_star, n0, xObs')) - ...
                0.5.*lamb_t.*om_star'*(Rd\om_star);
            


            logTau= -0.5.*lamb_y.*(yFld' - etaSim(temp, n0, xObs'))'* ...
                (yFld' - etaSim(temp, n0, xObs')) - ...
                0.5.*lamb_t.*om_temp'*(Rd\om_temp);
            
            
            logRatio= logTauStar - logTau;
            
            if exp(logRatio) >= 1

                om_temp= om_star;
                temp= exp(om_temp)./(1 + exp(om_temp));
                
                acceptCount= acceptCount + 1;
                
            else
                
                u= rand(1);
                if u <= exp(logRatio)
                    

                    om_temp= om_star;
                    temp= exp(om_temp)./(1 + exp(om_temp));
                    
                    acceptCount= acceptCount + 1;
                    
                end
                
                % Else, do nothing and leave tau alone;
                
            end  % End updating step on tau
            
        end  % end subIter loop
        om= om_temp;
        tau= exp(om)./(1 + exp(om));
        
        
        
        
        
        % Draw n0 using Metropolis proposals
        % Drawing on the complementary log-log scale
        m0= log(-1.*log(n0));
        
        m0_Star= m0 + cN0.*randn(1);
        
        
        log_m0Star= -0.5.*lamb_y.*((yFld' - etaSim(tau, exp(-1.*exp(m0_Star)), xObs'))' ...
            *(yFld' - etaSim(tau, exp(-1.*exp(m0_Star)), xObs'))) ...
            + m0_Star - exp(m0_Star);
        
        log_m0= -0.5.*lamb_y.*((yFld' - etaSim(tau, exp(-1.*exp(m0)), xObs'))' ...
            *(yFld' - etaSim(tau, exp(-1.*exp(m0)), xObs'))) ...
            + m0 - exp(m0);
        
        
        % check acceptance criterion
        logRatio= log_m0Star - log_m0;
        
        xcptProb= exp(logRatio);
        if xcptProb >= 1
            
            n0= exp(-1.*exp(m0_Star));
            acceptCountN0= acceptCountN0 + 1;
            
        else
            
            u= rand(1);
            if u <= xcptProb
                
                n0= exp(-1.*exp(m0_Star));
                acceptCountN0= acceptCountN0 + 1;
                
            end  % end if
            
        end  % end if
        
        
        
        
        
        % Draw lambda_tau
        
        a_lambStar= a_lamb + size(tau, 1)./2;
        b_lambStar= b_lamb + 0.5.*((om)')*(Rd\(om));
        
        lamb_t= gamrnd(a_lambStar, (b_lambStar).^(-1));
        
        % Let me know if the quadratic form is going negative or lamb_t
        % is NaN
        if (isnan(lamb_t) || ((tau-0.5)')*(Rd\(tau-0.5)) < 0)
            
            'uh-oh'
            
        end  % end error catch loop
        
        
        
        
        
        % Draw rho_t (correlation length) parameter. This uses Gaussian
        % proposals to draw on the complementary log-log scale (so that it is
        % the log scale suggested by Neal, 1998).
        nu_t= log(-1.*log(rho_t));
        
        nu_tStar= nu_t + cRho.*randn(1);
        
        
        rho_tStar= exp(-1.*exp(nu_tStar));
        
        
        % Find ratio - note that this involves inverting the corr matrix
        % for the candidate value of rho_t
        % Initialize and calculate upper-traingular of R
        Rstar= zeros(size(tau,1), size(tau,1));  % This one is for the Metropolis
        % step in sampling rho_t
        for i= 1:size(Rstar,1)
            
            for j= i:size(Rstar,2)
                
                Rstar(i,j)= rho_tStar^(4.*abs(xObs(i) - xObs(j)).^2);
                
            end
            
        end
        
        Rstar= Rstar + Rstar' - diag(diag(Rstar));
        
        % Add a small ridge (white noise) to the diagonal of R
        maxEig= max(eig(Rstar));
        cndNum= cond(Rstar);
        bnd= max([maxEig.*(cndNum - exp(20))./(cndNum.*(exp(20)-1)) 0]);
        
        delta= bnd;
        Rstard= Rstar + delta.*eye(size(Rstar,1), size(Rstar,1));
        

        [RstarRt, p]= chol(Rstard);
        detRstarInv_apx= -2.*sum(log(diag(RstarRt)));  
        
        [RRt, p]= chol(Rd);
        detRInv_apx= -2.*sum(log(diag(RRt)));
        
        

        log_rhotStar= 0.5.*detRstarInv_apx - 0.5.*lamb_t.*((om)')*(Rstard\(om)) ...
             + (b_tau - 1).*log(1 - rho_tStar) + nu_tStar - exp(nu_tStar);
        

        log_rhot= 0.5.*detRInv_apx - 0.5.*lamb_t.*((om)')*(Rd\(om)) ...
             + (b_tau - 1).*log(1 - rho_t) + nu_t - exp(nu_t);
        
        % Acceptance crieterion
        logRatio= log_rhotStar - log_rhot;
        
        xcptProb= exp(logRatio);
        if xcptProb >= 1
            
            rho_t= rho_tStar;
            acceptCountRho= acceptCountRho + 1;
            
        else
            
            u= rand(1);
            if u <= xcptProb
                
                rho_t= rho_tStar;
                acceptCountRho= acceptCountRho + 1;
                
            end  % end if
            
        end  % end if
        
        
        % Check to see if rho_t got set to 1 (somehow)
        if rho_t == 1
            
            'rho_t is 1!'
            break
            
        end  % end if
        
        
        
        % Adjust proposal distributions for tau and n0 every 100 itertions
        if (rem(iter, 100) == 0)
            
            if (acceptCount/(100.*5) > 0.25)
                
                c= 1.75.*c;
                
            else
                
                if (acceptCount/(100.*5) < 0.2)
                    
                    c= 0.25.*c;
                    
                end
                
            end
            
            
            if (acceptCountN0/100 > 0.5)
                
                cN0= 1.75.*cN0;
                
            else
                
                if (acceptCountN0/100 < .4)

                    cN0= 0.25.*cN0;
                    
                end
                
            end
            
            
            if (acceptCountRho/100 > 0.5)
                
                cRho= 1.75.*cRho;
                
            else
                
                if (acceptCountRho/100 < 0.4)
                    
                    cRho= 0.25.*cRho;
                    
                end
                
            end
            
            acceptCount= 0;
            acceptCountN0= 0;
            acceptCountRho= 0;
            
        end  % end if
        
        
        
        if (iter >= 10000)
            
            if (rem(iter,10) == 0)
                
                initTau(:, (iter./10)-999)= tau;  % hold the posterior draws of tau
                initN0((iter./10)-999)= n0;  % Recording draws on the complementary log-log scale
                initRhoT((iter./10)-999)= rho_t;  % Recording draws of rho_t on the
                % complementary log-log scale
                initLambY((iter./10)-999)= lamb_y;
                
            end  % end if
            
            
        end  % end if
        
        
    end % End burn-in loop
    
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    tauDraws= zeros(size(tau,1), 2000);
    n0Draws= zeros(1,2000);
    lamb_tDraws= zeros(1,2000);
    rho_tDraws= zeros(1, 2000);
    lamb_yDraws= zeros(1,2000);
    
    lkhd_Draws= zeros(1,2000);  % Store the evaluations of the log-lkhd
                                % for each iteration
    
                                
    % Sampling loop - basically the same as the previous loop, with no
    % adjustment of the proposal distributions
    for iter= 1:nsamples
        
        % Draw lamb_y
        a_yStar= a_y + size(tau, 1)./2;
        b_yStar= b_y + 0.5.*((yFld' - etaSim(tau, n0, xObs'))'* ...
             (yFld' - etaSim(tau, n0, xObs')));     
        lamb_y= gamrnd(a_yStar, (b_yStar).^(-1));

        
        
        
        % Draw tau vector
        % initialize and Calculate upper-traingular of R
        R= zeros(size(tau,1), size(tau,1));  % Initialize the correlation
        % matrix
        for i= 1:size(R,1)
            
            for j= i:size(R,2)
                
                R(i,j)= rho_t^(4.*abs(xObs(i) - xObs(j)).^2);
                
            end
            
        end
        
        R= R + R' - diag(diag(R));  % Fill in lower traingular of R
        

        maxEig= max(eig(R));
        cndNum= cond(R);
        bnd= max([maxEig.*(cndNum - exp(20))./(cndNum.*(exp(20)-1)) 0]);
        
        delta= bnd;
        Rd= R + delta.*eye(size(R,1), size(R,1));
        

        
        

        [U, D]= eig((lamb_t).^(-1).*R);  
        
        %         if p ~= 0
        %
        %             'problem with R'
        %             break
        %
        %         end
        
        
        
        % Neal (1998) suggested doing several iterations of the GP draws for
        % the Metropolis-within-Gibbs step to improve convergence
        om_temp= om;
        temp= exp(om)./(1 + exp(om));
        for subIter= 1:5
            

            om_star= c.*U*diag(sqrt(diag(D)))*randn(size(om)) + om_temp;
            om_star= real(om_star);
            
            

            
            
            tau_star= exp(om_star)./(1 + exp(om_star));
            


            logTauStar= -0.5.*lamb_y.*(yFld' - etaSim(tau_star, n0, xObs'))'* ...
                (yFld' - etaSim(tau_star, n0, xObs')) - ...
                0.5.*lamb_t.*om_star'*(Rd\om_star);
            
            


            logTau= -0.5.*lamb_y.*(yFld' - etaSim(temp, n0, xObs'))'* ...
                (yFld' - etaSim(temp, n0, xObs')) - ...
                0.5.*lamb_t.*om_temp'*(Rd\om_temp);
            
            
            logRatio= logTauStar - logTau;
            
            if exp(logRatio) >= 1
                

                om_temp= om_star;
                temp= exp(om_temp)./(1 + exp(om_temp));
                
                acceptCount= acceptCount + 1;
                
            else
                
                u= rand(1);
                if u <= exp(logRatio)
                    

                    om_temp= om_star;
                    temp= exp(om_temp)./(1 + exp(om_temp));
                    
                    acceptCount= acceptCount + 1;
                    
                end
                
                % Else, do nothing and leave tau alone;
                
            end  % End updating step on tau
            
        end  % end subIter loop
        om= om_temp;
        tau= exp(om)./(1 + exp(om));
        
        
        
        
        
        % Draw n0 using Metropolis proposals
        % Drawing on the complementary log-log scale
         m0= log(-1.*log(n0));
        
        m0_Star= m0 + cN0.*randn(1);

        
        log_m0Star= -0.5.*lamb_y.*((yFld' - etaSim(tau, exp(-1.*exp(m0_Star)), xObs'))' ...
            *(yFld' - etaSim(tau, exp(-1.*exp(m0_Star)), xObs'))) ...
            + m0_Star - exp(m0_Star);
        
        log_m0= -0.5.*lamb_y.*((yFld' - etaSim(tau, exp(-1.*exp(m0)), xObs'))' ...
            *(yFld' - etaSim(tau, exp(-1.*exp(m0)), xObs'))) ...
            + m0 - exp(m0);
        
        
        % check acceptance criterion
        logRatio= log_m0Star - log_m0;
        
        xcptProb= exp(logRatio);
        if xcptProb >= 1
            
            n0= exp(-1.*exp(m0_Star));
            acceptCountN0= acceptCountN0 + 1;
            
        else
            
            u= rand(1);
            if u <= xcptProb
                
                n0= exp(-1.*exp(m0_Star));
                acceptCountN0= acceptCountN0 + 1;
                
            end  % end if
            
        end  % end if
        
        
        
        
        
        % Draw lambda_tau
        
        a_lambStar= a_lamb + size(tau, 1)./2;
        b_lambStar= b_lamb + 0.5.*((om)')*(Rd\(om));
        
        lamb_t= gamrnd(a_lambStar, (b_lambStar).^(-1));
        
        % Let me know if the quadratic form is going negative or lamb_t
        % is NaN
        if (isnan(lamb_t) || ((tau-0.5)')*(Rd\(tau-0.5)) < 0)
            
            'uh-oh'
            
        end  % end error catch loop
        
        
        
        
        
        % Draw rho_t (correlation length) parameter.
        nu_t= log(-1.*log(rho_t));
        
        nu_tStar= nu_t + cRho.*randn(1);
        rho_tStar= exp(-1.*exp(nu_tStar));
        
        
        % Find ratio - note that this involves inverting the corr matrix
        % for the candidate value of rho_t
        % Initialize and calculate upper-traingular of R
        Rstar= zeros(size(tau,1), size(tau,1));  % This one is for the Metropolis
        % step in sampling rho_t
        for i= 1:size(Rstar,1)
            
            for j= i:size(Rstar,2)
                
                Rstar(i,j)= rho_tStar^(4.*abs(xObs(i) - xObs(j)).^2);
                
            end
            
        end
        
        Rstar= Rstar + Rstar' - diag(diag(Rstar));
        
        % Add a small ridge (white noise) to the diagonal of R
        maxEig= max(eig(Rstar));
        cndNum= cond(Rstar);
        bnd= max([maxEig.*(cndNum - exp(20))./(cndNum.*(exp(20)-1)) 0]);
        
        delta= bnd;
        Rstard= Rstar + delta.*eye(size(Rstar,1), size(Rstar,1));
        

        [RstarRt, p]= chol(Rstard);
        detRstarInv_apx= -2.*sum(log(diag(RstarRt)));  % Note that these are
        % already on the log scale
        
        [RRt, p]= chol(Rd);
        detRInv_apx= -2.*sum(log(diag(RRt)));
        
        

        log_rhotStar= 0.5.*detRstarInv_apx - 0.5.*lamb_t.*((om)')*(Rstard\(om)) ...
             + (b_tau - 1).*log(1 - rho_tStar) + nu_tStar - exp(nu_tStar);
        

        log_rhot= 0.5.*detRInv_apx - 0.5.*lamb_t.*((om)')*(Rd\(om)) ...
             + (b_tau - 1).*log(1 - rho_t) + nu_t - exp(nu_t);
        
        % Acceptance crieterion
        logRatio= log_rhotStar - log_rhot;
        
        xcptProb= exp(logRatio);
        if xcptProb >= 1
            
            rho_t= rho_tStar;
            acceptCountRho= acceptCountRho + 1;
            
        else
            
            u= rand(1);
            if u <= xcptProb
                
                rho_t= rho_tStar;
                acceptCountRho= acceptCountRho + 1;
                
            end  % end if
            
        end  % end if
        
        
        % Check to see if rho_t got set to 1 (somehow)
        if rho_t == 1
            
            'rho_t is 1!'
            break
            
        end  % end if
        
        
        
        
        
        if(rem(iter, 2) == 0)
            
            tauDraws(:, iter./2)= tau;
            n0Draws(iter./2)= n0;
            lamb_tDraws(iter./2)= lamb_t;
            rho_tDraws(iter./2)= rho_t;
            lamb_yDraws(iter./2)= lamb_y;
            
            % Evaluate the log-lkhd
            R= zeros(size(tau,1), size(tau,1));  
            for i= 1:size(R,1)

                for j= i:size(R,2)

                    R(i,j)= rho_t^(4.*abs(xObs(i) - xObs(j)).^2);

                end

            end

            R= R + R' - diag(diag(R));  % Fill in lower traingular of R


            maxEig= max(eig(R));
            cndNum= cond(R);
            bnd= max([maxEig.*(cndNum - exp(20))./(cndNum.*(exp(20)-1)) 0]);

            delta= bnd;
            Rd= R + delta.*eye(size(R,1), size(R,1));
            
            [RRt, p]= chol(Rd);
            detRInv_apx= -2.*sum(log(diag(RRt)));
            
            
            logLkhd= 0.5.*log(lamb_y) - 0.5.*lamb_y.*(yFld' - etaSim(tau_star, n0, xObs'))'* ...
                (yFld' - etaSim(tau_star, n0, xObs')) + (a_y - 1).*log(lamb_y) - b_y.*lamb_y + ...
                (size(tau, 1)./2).*lamb_t.*detRInv_apx - 0.5.*lamb_t.*((om)')*(Rd\(om)) + ...
                (b_tau - 1).*log(1 - rho_t).*(a_lamb - 1).*log(lamb_t) - b_lamb.*lamb_t;
            
            lkhd_Draws(iter./2)= logLkhd;
            
        end  % end if
        
    end % End sampling loop
    
    TauDraws{chain_count} = tauDraws;
    N0Draws{chain_count} = n0Draws;
    Lamb_tDraws{chain_count} = lamb_tDraws;
    Rho_tDraws{chain_count} = rho_tDraws;
    Lamb_yDraws{chain_count} = lamb_yDraws;
    Lkhd_Draws{chain_count} = lkhd_Draws;
    
end

matlabpool close


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% The objects TauDraws{1}, TauDraws{2}, etc. store the sample draws for
% each chain for each parameter. These can be used to assess covergence
% through trace plots, scale reduction factors, etc. Upon attaining
% approximate convergence, combine the three chains for one Monte Carlo
% sample of size 6,000 from the approximate joint posterior distribution.





